%[text] # Modify User Behavior
%[text] ![](text:image:0d53) Use the calculator to convert from seconds to minutes (to the nearest hundredth).
NumberOfSeconds = 60;     %[control:editfield:42fa]{"position":[19,21]} %[control:button:12c9]{"position":[25,26]}
TimeInMinutes = SecondsToMinutes(NumberOfSeconds);
function minutes = SecondsToMinutes(seconds)
minutes = seconds/60; % Convert seconds to minutes
end
if TimeInMinutes == round(TimeInMinutes,2)
    fprintf("%d seconds is %.2f minutes.",NumberOfSeconds,TimeInMinutes)
else
    fprintf("%d seconds is approximately %.2f minutes.",NumberOfSeconds,TimeInMinutes)
end
%%
%[text] ## Set Up Your Timer Algorithm
Alert.Algorithm = "Total time"; %[control:dropdown:62b7]{"position":[19,31]}
%[text] 
%[text] **1. When the user enters text neck, wait**
Alert.Time = 1*10; %[control:editfield:630c]{"position":[14,15]}
%[text] **before alerting.**
%[text] 
%[text] **2. Alert the user for**
Alert.OffTime = 1.05*10; %[control:editfield:9037]{"position":[17,21]}
%[text] 
%[text] **3. Before sending another alert, wait**
Alert.RestartTime = 3*10; %[control:editfield:1a62]{"position":[21,22]}
%[text] 
%[text] ## Set Up the User's Behavior 
%[text] 
%[text] **The user will stay out of text neck for** 
Alert.TotalSkipTime = 10*10 ; % 10 minutes in 0.1 min steps %[control:editfield:42de]{"position":[23,25]}
%[text] **after reacting to an alert.** 
%[text] 
%[text] Set the probability of the user going out of text neck with each alert.
AlarmPercents = [25, ... %[control:slider:235a]{"position":[18,20]}
    50, ... %[control:slider:7484]{"position":[5,7]}
    75, ... %[control:slider:4a68]{"position":[5,7]}
    100]; %[control:slider:6949]{"position":[5,8]}
Alert.AlarmProbability = AlarmPercents/100;
% Additional Alert settings
MaxTime = 1200; % 120 minutes in 0.1 min steps
% This is the length of time that reacting to an alert keeps a user from
% re-entering text neck.

%[text] 
%[text] Use the button below to submit your settings:
  %[control:button:3a83]{"position":[1,2]}
InputChecks = [CheckInputs(Alert.Time,MaxTime); CheckInputs(Alert.OffTime,MaxTime);CheckInputs(Alert.RestartTime,MaxTime)];
function Results = CheckInputs(Val,MaxVal)
if Val < 0
    NegVal = true;
    OverMaxVal = false;
elseif Val > MaxVal
    NegVal = false;
    OverMaxVal = true;
else
    NegVal = false;
    OverMaxVal = false;
end
Results = [NegVal OverMaxVal];
end
if any(InputChecks(:, 1))
    disp("Negative values do not make sense. Please use non-negative values.")
elseif any(InputChecks(:,2))
    disp("Please limit your alarm values to " + MaxTime/100 + " minutes.")
else
    disp("You set up the following alert algorithm:")
    fprintf("  Method: %s \n" + ...
        "  In text neck: %.2f minutes\n" + ...
        "  Alert stays on: %.2f minutes\n" + ...
        "  Between alerts: %.2f minutes\n" + ...
        "  Remain alerted: %.2f minutes\n" + ...
        "  With response probabilities: %.0f%% to the first alert, %.0f%% to the second, %.0f%% to the third, and %.0f%% to the fourth.",...
        Alert.Algorithm,Alert.Time/10, ...
        Alert.OffTime/10,Alert.RestartTime/10,Alert.TotalSkipTime/10,Alert.AlarmProbability(1)*100,...
        Alert.AlarmProbability(2)*100,Alert.AlarmProbability(3)*100,Alert.AlarmProbability(4)*100)
end
%%
%[text] ## Test Your Algorithm
%%
%[text] **First, see what the user does without any alerts.**
seed = 0; %[control:editfield:0d78]{"position":[8,9]}
  %[control:button:9ec4]{"position":[1,2]}

if seed < 0
    disp("Please enter a number that is not negative.")
    seed = 0;
elseif seed > 2^32
    disp("Please enter a number that is smaller than 2^32")
    seed = 2^32-1;
elseif seed ~= round(seed)
    disp("You should enter a whole number.")
    seed = round(seed);
end

fprintf("Please wait, simulation running...");

function Alert = InitializeAlertSettings(Alert,MaxTime)
Alert.History = false([1 MaxTime]);
Alert.React = [];
Alert.Number = 0; % Count of Alerts
% Alert.AlarmProbability = [0.25 0.5 0.75 1];
Alert.TimeInTNP = false([1 MaxTime]);
end

if exist("Alert","var")
    Alert = InitializeAlertSettings(Alert,MaxTime);
NoAlert = Alert;
NoAlert.Algorithm = "None";
TotTimeSteps = 1200;
% Angles = CreateRandomScenario(1200); % Random scenario of 120 minutes
    [~,BaseData,~] = CreateRandomScenario(TotTimeSteps,NoAlert,seed,IncludeFigure=false);
else
    warning("You must submit an algorithm before running a simulation.")
end    
    
fprintf("Done!");

f = figure('Position', [0 0 5000 1000]);
    plot(BaseData.History, LineWidth=2)
    xticks(0:100:TotTimeSteps)
    xticklabels(0:10:TotTimeSteps/10)
    xlabel("Time (minutes)")
    ylabel("Text Neck?")
    ylim([-0.25 1.25])
    yticks([0 1])
    yticklabels(["No" "Yes"])
    title("Text Neck over Time with No Alarms")
%    [TNPlot,TNData,Alert] = CreateRandomScenario(1200,Alert,seed,IncludeFigure=true);

%%
%[text] **Next, see how the user’s behavior changes when you add alerts.**
  %[control:button:6979]{"position":[1,2]}
fprintf("Please wait, simulation running...");

% Angles = CreateRandomScenario(1200); % Random scenario of 120 minutes
if any(Alert.History==1)
Alert = InitializeAlertSettings(Alert,MaxTime);
end
[TNPlot,TNData,Alert] = CreateRandomScenario(1200,Alert,seed,IncludeFigure=false);
%%
fprintf("Done!");
%plot figure
GenerateSummaryPlot(TNPlot,TNData,Alert)
PrintSummaryStatistics(BaseData,TNData,Alert)
%%


% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% % Initialize data structure 
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function TNData = InitializeTextNeckData(TotTimeSteps)
TNData.TextNeck = false;
TNData.History = zeros([1 TotTimeSteps]);
end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% % Create Random Scenario
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [TNPlot,TNData,Alert] = CreateRandomScenario(TotTimeSteps,Alert,seed,opts)
arguments
    TotTimeSteps 
    Alert 
    seed = 1
    opts.IncludeFigure = true 
end
rng(seed)

TNPlot = SetUpPlots(TotTimeSteps,opts.IncludeFigure); % With step size 0.1 min
    
TNData = InitializeTextNeckData(TotTimeSteps);
%AngleHistory = zeros([1 TotTimeSteps]);
direction = randi(2);
speed = 0;

for timeStep = 1:1200 % in 0.1 minute increments
    % if Alert.React(timeStep) > 0 && Alert.React(timeStep) < Alert.TotalSkipTime
    %     Alert.React(timeStep+1) = Alert.React(timeStep)+1;
    % elseif Alert.React(timeStep) == Alert.TotalSkipTime
    %     Alert.React(timeStep+1) = 0;
    % if ~isempty(Alert.React) && timeStep - Alert.React(end) < Alert.TotalSkipTime
        
    if ShouldChangeDirection(timeStep)
        direction = randi(2);
        speed = GenerateRandomSpeed();
    end
    % Move the head down or up based on the mode
    [TNPlot,TNData,Alert] = CheckPosition(timeStep, TNPlot, TNData, Alert,opts.IncludeFigure);
    if ~isempty(Alert.React) && timeStep - Alert.React(end) < Alert.TotalSkipTime
        % Move with maximal speed back towards neutral
        MoveTowardNeutral(TNPlot,3,TNData.CurAngle);
    else
        ApplyHeadRotation(TNPlot, direction, speed, TNData.CurAngle)
    end
    %    AngleHistory(timeStep) = TNData.CurAngle;
end
end

function [TextNeck,angle] = DetermineNeckPosition(TNPlot,TNAngle)
arguments
    TNPlot
    TNAngle (1,1) double {mustBeInteger,mustBePositive} = 20
end

angle = CalculateHeadAngle(TNPlot);

% Compute TextNeck values for the current position
if abs(angle) < (180-TNAngle)
    % Set TextNeck=true if the neck is bent more than 20 degrees
    TextNeck = true;
    % If the neck angle moves within 20 degrees of 180
elseif abs(angle) >= (180-TNAngle)
    TextNeck = false;
end
end

function [TNPlot,TNData,Alert] = CheckPosition(timeStep, TNPlot, TNData, Alert, DrawPlots)
% CHECKPOSITION Function to check the position of the neck
%
% Input Arguments:
%     i      - Current timestep index (0.01 minutes each)
%     TNPlot - Structure containing
%        CurrPlot - a line plot of the textneck status no/0 vs yes/1
%        SensorPlot - a scatter showing where the sensor is located (at the
%              head/neck point on the figure)
%        TextNeckAxes - a handle to the axes containing CurrPlot
%        head - a handle to the line plot of the figure's head
%        torso - a handle to the line plot of the figure's torso
%     TNData - Structure containing data related to neck position
%        ContiguousTimeInTNP - the time in TextNeck without leaving
%        TextNeck - the current state, true is in TextNeck position
%        History - a yes/no record for TextNeck position
%     Alert   - Structure containing alert timing parameters
%        Time - The amount of time in TextNeck that will set off an alarm
%        RestartTime - The length of time after the alarm turns off before
%             the alarm will go off again while a user persists in TextNeck
%        History - an on/off record for the alerts
%        Algorithm - either "Total time" or "Continuous time"
%        React - a time count when the user is reacting to an alarm
%        Number - a count of the total number of alerts that have occurred
%        in this scenario
%        TotalSkipTime - the number of timesteps the user should avoid
%        entering text-neck after reacting to an alert
%      DrawPlots - a logical indicating whether TNPlot is drawn or not
% Output Arguments:
%     TNPlot - Updated plot data structure
%     TNData - Updated data structure with TextNeck History
%     Alert - Updated alert structure

% Update TNData value at time i

% First, calculate whether we are in TextNeck at all
[TNData.TextNeck,TNData.CurAngle] = DetermineNeckPosition(TNPlot);
TNData.History(timeStep) = double(TNData.TextNeck);

[Alert,TNData] = ShouldAlarm(timeStep,TNData,Alert);
[TNPlot,Alert] = UpdatePlotsWithAlarms(timeStep,TNPlot,TNData,Alert,DrawPlots);

end

function GenerateSummaryPlot(TNPlot,TNData,Alert)
% GENERATESUMMARYPLOT Create final summary plot of text neck over time
%
% Input Arguments:
%   numTimeSteps - Total number of time steps simulated
%   AllPlots - Array of plot handles to copy into summary figure

% Create new figure with wide aspect ratio for time series
f = figure('Position', [0 0 5000 1000]);
a = axes(f);
numTimeSteps = numel(TNData.History);
plot(TNData.History,LineWidth=2)

% Plot TextNeck History
hold on
for k = 1:length(TNPlot.AllPlots)
    copyobj(TNPlot.AllPlots(k), a);
end
if ~isempty(Alert.React)
xline(Alert.React,"g",LineWidth=0.5,Alpha=0.2)
for StartReactTime = Alert.React
    patch([StartReactTime StartReactTime StartReactTime+Alert.TotalSkipTime StartReactTime+Alert.TotalSkipTime], [-0.25 1.25 1.25 -0.25], "g", EdgeColor="none", FaceAlpha=0.2, Parent=a);
end
end
hold off

% Configure axes
xlim([0 numTimeSteps]);
ylim([-0.25 1.25]);
xlabel('Time (minutes)');
ylabel('Text Neck?');
title('Text Neck Over Time With Alerts');

% Set up ticks (assuming 10 time steps = 1 minute)
yticks([0 1]);
yticklabels({'No', 'Yes'});
xticks(0:100:numTimeSteps);
xticklabels(0:10:(numTimeSteps/10));
xtickangle(0);
end


function PrintSummaryStatistics(BaseData,TNData,Alert)
% PRINTSUMMARYSTATISTICS Display summary statistics to console
%
% Input Arguments:
%   TNData.history contains timestamped logical yes/no of text neck history
%
% Notes:
%   Assumes 10 time steps = 1 minute for time conversion

UserResponse = numel(Alert.React);

totalBaseMinutes = sum(BaseData.History)/10;
totalBaseEntries = sum(diff(BaseData.History)==1);
totalMinutes = sum(TNData.History) / 10;
totalEntries = sum(diff(TNData.History) == 1);

% Create a table to display the summary statistics
summaryTable = table( ...
    [totalBaseEntries; totalBaseMinutes], ...
    [totalEntries; totalMinutes], ...
    'VariableNames', {'User Behavior With No Alerts', 'User Behavior With Alerts'},...
    'RowNames',{'Number of times in text neck','Total time in text neck '});
disp(summaryTable)

% fprintf('Number of times entering text neck position originally: %i and with alerts: %i \n', totalBaseEntries,totalEntries);
fprintf("The user responded to %i alerts and ignored %i alerts.\n", UserResponse,Alert.Number-UserResponse)
% fprintf('Total time in text neck position originally: %.1f minutes and with alerts: %.1f minutes \n', totalBaseMinutes,totalMinutes);
end

function speed = GenerateRandomSpeed()
% GENERATERANDOMSPEED Generate random rotation speed between 1 and 3 degrees/step
speed = 1 + (3 - 1) * rand;
end


function shouldChange = ShouldChangeDirection(timeStep)
% SHOULDCHANGEDIRECTION Determine if direction should change at current time step
%
% Direction changes every 10 time steps by checking if the tens digit changes
shouldChange = mod(floor(timeStep/10), 2) ~= mod(floor((timeStep-1)/10), 2);
end


function angle = CalculateHeadAngle(TNPlot)
% CALCULATEHEADANGLE Calculate angle between head and torso orientation vectors
%
% Output Arguments:
%   Vec1 - Vector from torso reference point to head position
%   Vec2 - Vector representing torso orientation (from point 1 to point 2)
%   angle - Angle between Vec1 and Vec2 in degrees

head = TNPlot.head;
torso = TNPlot.torso;
% Extract key points from head and torso
headPoint = [head.XData(250), head.YData(250), head.ZData(250)];
torsoReferencePoint = [torso.XData(2), torso.YData(2), torso.ZData(2)];
torsoTopPoint = [torso.XData(1), torso.YData(1), torso.ZData(1)];

% Calculate vectors
Vec1 = headPoint - torsoReferencePoint;
Vec2 = torsoTopPoint - torsoReferencePoint;

% Calculate angle using vector cross product and dot product
angle = atan2d(norm(cross(Vec1, Vec2)), dot(Vec1, Vec2));
end

function MoveTowardNeutral(TNPlot,speed,currentAngle)
    if abs(currentAngle)+speed > 180
        return
    else
        ApplyHeadRotation(TNPlot, 2, speed, currentAngle)
    end
end

function ApplyHeadRotation(TNPlot, direction, speed, currentAngle)
% APPLYHEADROTATION Rotate head based on direction with angle constraints
%
% Input Arguments:
%   head - Handle to head graphics object
%   torso - Handle to torso graphics object
%   direction - Rotation direction (1 = down/forward, 2 = up/back)
%   speed - Rotation speed in degrees
%   currentAngle - Current angle between head and torso in degrees
%
% Notes:
%   - Direction 1 (down): Only rotate if angle >= 120° (prevents over-rotation)
%   - Direction 2 (up): Only rotate if angle <= 177° (prevents hyper-extension)
head = TNPlot.head;
torso = TNPlot.torso;

% Get rotation pivot point (torso reference point)
pivotPoint = [torso.XData(2), torso.YData(2), torso.ZData(2)];

if direction == 1
    % Rotate head forward/down (around positive x-axis)
    if currentAngle >= 120
        rotate(head, [1 0 0], speed, pivotPoint);
    end
elseif direction == 2
    % Rotate head backward/up (around negative x-axis)
    if currentAngle <= 177
        rotate(head, [-1 0 0], speed, pivotPoint);
    end
end
end

function [Alert,TNData] = ShouldAlarm(timeStep,TNData,Alert)
if Alert.Algorithm == "Total time"
    % The Total Time Algorithm says that we should alert if
    %    In TextNeck position AND (sum(TNData.History) > Alert.Time)
    %       AND (mod(sum(TNData.History)-Alert.Time,RestartTime+OffTime) < OffTime)

    % Check whether an alert is triggered
    if TNData.TextNeck && sum(TNData.History) >= Alert.Time && ...
            (mod(sum(TNData.History)-Alert.Time,Alert.OffTime+Alert.RestartTime) < Alert.OffTime)
        Alert.History(timeStep) = true;
    else
        Alert.History(timeStep) = false;
    end
elseif Alert.Algorithm == "Continuous time"
    % Increment contiguous time in TextNeck if currently in that state or reset
    % it to 0 if we are not currently in TextNeck
    if TNData.TextNeck == true
        TNData.ContiguousTimeInTNP = TNData.ContiguousTimeInTNP + 1;
    else
        TNData.ContiguousTimeInTNP = 0;
    end

    % The Continuous Time Algorithm says that we should alert if
    %    In TextNeck position AND (ContiguousTimeInTNP > Alert.Time)
    %       AND (mod(ContiguousTimeInTNP-Alert.Time,RestartTime+OffTime) < OffTime)

    % Check whether an alert is triggered
    if TNData.TextNeck && TNData.ContiguousTimeInTNP >= Alert.Time && ...
            (mod(TNData.ContiguousTimeInTNP-Alert.Time,Alert.OffTime+Alert.RestartTime) < Alert.OffTime)
        Alert.History(timeStep) = true;
    else
        Alert.History(timeStep) = false;
    end
elseif Alert.Algorithm == "None"
    Alert.History(timeStep) = false;
else
    warning("Unexpected alert algorithm: " + Alert.Algorithm)
end

end



function Alert = CheckReaction(timeStep,Alert)
% Check whether the the user reacts to this alert based on the
% probabilities set in Alert.AlarmProbability
% Append the current timeStep to Alert.React if the user reacts
assert(Alert.Number > 0)
idx = mod(Alert.Number-1,4)+1;
Chance = Alert.AlarmProbability(idx);
s = rng;             % capture current RNG state (structure)
save("rngState.mat","s")
rng("shuffle");
roll = rand(1);
load("rngState.mat","s")
rng(s)   
if Chance >= roll
    Alert.React = [Alert.React timeStep];
%    disp("Success! The user reacted to your alert at " + timeStep/10 + " minutes!")
%else
    % Alert.React(timeStep) = 0;
    % disp("The user ignored your alert at " + timeStep/10 + " minutes.")
end
end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% % Create Plots
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [TNPlot,Alert] = UpdatePlotsWithAlarms(timeStep,TNPlot,TNData,Alert,DrawPlots)
% Update current plot data with new index and TextNeck status
if Alert.History(timeStep)
    assert(timeStep>1)
    if (Alert.History(timeStep)-Alert.History(timeStep-1))==1
        % When an alert first turns on, record the new alert and determine
        % whether the user will react
        Alert.Number = Alert.Number + 1;
        Alert = CheckReaction(timeStep,Alert);
    end
    % Turn the sensor icon on
    TNPlot.SensorPlot.MarkerFaceColor = 'r';
    if ~isempty(Alert.React) && timeStep - Alert.React(end) < Alert.TotalSkipTime
        if Alert.React(end) == timeStep
            % Add the initial reaction line in green
            xline(timeStep,LineWidth=1,Color="g",Alpha=0.4)
        end
        % Extend green shading over this timestep
        hold on
        TNPlot.CurrPlot = plot([timeStep-1 timeStep], TNData.History(timeStep-1:timeStep), 'g', 'Parent', TNPlot.TextNeckAxes);
        p = patch([timeStep-1 timeStep-1 timeStep timeStep], [-0.25 1.25 1.25 -0.25], 'g', 'EdgeColor', 'none', 'FaceAlpha', 0.2, 'Parent', TNPlot.TextNeckAxes);
        TNPlot.AllPlots = [TNPlot.AllPlots p];
        hold off
    else
        % Extend red shading over this timestep
        hold on
        TNPlot.CurrPlot = plot([timeStep-1 timeStep], TNData.History(timeStep-1:timeStep), 'r', 'Parent', TNPlot.TextNeckAxes);
        p = patch([timeStep-1 timeStep-1 timeStep timeStep], [-0.25 1.25 1.25 -0.25], 'red', 'EdgeColor', 'none', 'FaceAlpha', 0.2, 'Parent', TNPlot.TextNeckAxes);
        TNPlot.AllPlots = [TNPlot.AllPlots p];
        hold off
    end
else
    % Turn the sensor icon off
    TNPlot.SensorPlot.MarkerFaceColor = 'none';
    if ~isempty(Alert.React) && timeStep - Alert.React(end) < Alert.TotalSkipTime
        % Switch line to green
        hold on
        TNPlot.CurrPlot = plot([timeStep-1 timeStep], TNData.History(timeStep-1:timeStep), 'g', 'Parent', TNPlot.TextNeckAxes);
        hold off
    elseif timeStep == 1
        hold on
        TNPlot.CurrPlot = plot(timeStep-1,0,"b",Parent=TNPlot.TextNeckAxes);
        hold off
    else
        % Switch line back to blue
        hold on
        TNPlot.CurrPlot = plot([timeStep-1 timeStep], TNData.History(timeStep-1:timeStep), 'b', 'Parent', TNPlot.TextNeckAxes);
        hold off
    end
end


if DrawPlots && mod(timeStep,10) == 0
% Ensure the plot is updated with the latest data
    pause(0.05);
end
end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% % Create Plots
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function TextNeckPlots = SetUpPlots(TotTimeSteps,ShowPlots)
arguments
    TotTimeSteps (1,1) uint16 = 1200
    ShowPlots (1,1) logical = true
end
% SETUPPLOTS Function to set up 3D plots for torso and head, and a 2D plot for text neck over time
%
% Output Arguments:
%     TextNeckPlots - structure containing handles for the plots
%         .CurrPlot - handle for the current plot of text neck over time
%         .SensorPlot - handle for the sensor plot
%         .TextNeckAxes - axes for the text neck plot
%         .head - handle for the head plot
%         .torso - handle for the torso plot

clf % Clear current figure
if ShowPlots
    f = figure('Position', [0 0 1100 400]); % Create a new figure with specified position
else
    f = figure('Position',[0 0 1100 400],'Visible',"off");
end
t = tiledlayout(1,2); % Create a tiled layout for multiple plots
nexttile; % Move to the next tile in the layout
% Plot torso as a 3D line
torso = plot3([4 4], [4 4], [4 8], 'SeriesIndex', 1, 'LineWidth', 3);
hold on % Hold the current plot to overlay additional plots
% Plot arms as a 3D line
arms = plot3([3 2.5 3 4 5 5.5 5], zeros(1, 7) + 4, [5 6 7 7 7 6 5], 'SeriesIndex', 1, 'LineWidth', 3);
% Plot legs as a 3D line
legs = plot3([3 3 3 4 5 5 5], [3.5, zeros(1, 5) + 4, 3.5], [1 1 3 4 3 1 1], 'SeriesIndex', 1, 'LineWidth', 3);
theta = linspace(0, 2*pi, 1000); % Create an array of angles for the head
x = 0.5 * cos(theta) + 4; % X coordinates for the head
y = zeros(1, length(theta)) + 4; % Y coordinates for the head
z = sin(theta) + 9; % Z coordinates for the head
% Plot head as a 3D line
head = plot3(x, y, z, 'SeriesIndex', 1, 'LineWidth', 3);
MyColors = colororder; % Get the current color order
% Fill areas with transparency to represent different sections
fill3([0 0 10 10], [0 10 10 0], [1 1 1 1], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
fill3([10 10 10 10], [0 10 10 0], [1 1 10 10], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
fill3([0 0 10 10], [10 10 10 10], [1 10 10 1], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
xlim([0 10]) % Set limits for the x-axis
ylim([0 10]) % Set limits for the y-axis
zlim([0 11]) % Set limits for the z-axis
hold off
ax = gca; % Get current axes
ax.Visible = 'off'; % Hide the axes
SensorPlot = PlotSensor(torso); % Call function to plot sensor data

% Set up the axes for the text neck plot
TextNeckAxes = axes(t);
TextNeckAxes.Layout.Tile = 2;   % Position the axes in the second tile
CurrPlot = plot(0, false, 'b'); % Initialize the current plot
xlim([0 TotTimeSteps]);         % Set limits of the text neck plot in 1/100 minute increments
ylim([-0.25 1.25]);             % Set limits of the text neck plot
xlabel('Time (minutes)');              % Label for the x-axis
ylabel('Text Neck?');           % Label for the y-axis
title('Text Neck Over Time');   % Title for the plot
yticks([0 1]);                  % Set y-ticks
xticks(0:100:TotTimeSteps);     % Set x-ticks
xticklabels(0:10:TotTimeSteps/10); % Set x-tick labels
xtickangle(0);                  % Set angle for x-tick labels
yticklabels(["No" "Yes"]);      % Set y-tick labels

% Create output structure
TextNeckPlots.CurrPlot = CurrPlot;
TextNeckPlots.AllPlots = CurrPlot;
TextNeckPlots.SensorPlot = SensorPlot;
TextNeckPlots.TextNeckAxes = TextNeckAxes;
TextNeckPlots.head = head;
TextNeckPlots.torso = torso;
end

function SensorPlot = PlotSensor(torso)
hold on
SensorPlot = scatter3(torso.XData(2), torso.YData(2), torso.ZData(2), 'SizeData', 30, 'SeriesIndex', 'none');
hold off

ax = gca;
ax.Visible = 'off';
end

%[appendix]{"version":"1.0"}
%---
%[metadata:view]
%   data: {"layout":"hidecode","rightPanelPercent":40}
%---
%[text:image:0d53]
%   data: {"align":"baseline","height":31,"src":"data:image\/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4KPHN2ZyBpZD0iTGF5ZXJfMSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB2ZXJzaW9uPSIxLjEiIHZpZXdCb3g9IjAgMCA4MC45MiA5OS42OSI+CiAgPCEtLSBHZW5lcmF0b3I6IEFkb2JlIElsbHVzdHJhdG9yIDMwLjAuMCwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDIuMS4xIEJ1aWxkIDEyMykgIC0tPgogIDxkZWZzPgogICAgPHN0eWxlPgogICAgICAuc3QwIHsKICAgICAgICBmaWxsOiAjMTA0ODcxOwogICAgICB9CgogICAgICAuc3QxIHsKICAgICAgICBmaWxsOiAjZmZmOwogICAgICB9CgogICAgICAuc3QyIHsKICAgICAgICBmaWxsOiAjYjBkMGU0OwogICAgICB9CgogICAgICAuc3QzIHsKICAgICAgICBmaWxsOiAjZGNlYmYzOwogICAgICB9CgogICAgICAuc3Q0IHsKICAgICAgICBmaWxsOiAjNWFiZmExOwogICAgICB9CgogICAgICAuc3Q1IHsKICAgICAgICBmaWxsOiAjNWI4NWEzOwogICAgICB9CgogICAgICAuc3Q2IHsKICAgICAgICBmaWxsOiAjYjQzNTM2OwogICAgICB9CiAgICA8L3N0eWxlPgogIDwvZGVmcz4KICA8cGF0aCBjbGFzcz0ic3Q1IiBkPSJNNjguNDQsOTUuOThIMTIuNjRjLTQuMiwwLTcuNy0zLjQtNy43LTcuN1YxMi4zOGMwLTQuMiwzLjQtNy43LDcuNy03LjdoNTUuOGM0LjIsMCw3LjcsMy40LDcuNyw3Ljd2NzZjMCw0LjItMy41LDcuNi03LjcsNy42WiIvPgogIDxwYXRoIGNsYXNzPSJzdDMiIGQ9Ik02Ny4zNCwyNy4xOEgxMy43NGMtMS43LDAtMy4xLTEuNC0zLjEtMy4xdi05LjRjMC0xLjcsMS40LTMuMSwzLjEtMy4xaDUzLjZjMS43LDAsMy4xLDEuNCwzLjEsMy4xdjkuNGMwLDEuOC0xLjQsMy4xLTMuMSwzLjFaIi8+CiAgPHBvbHlnb24gY2xhc3M9InN0MSIgcG9pbnRzPSI1Ni40NCAyNy4xOCAzNy42NCAyNy4xOCAyNC42NCAxMS43OCA0My40NCAxMS43OCA1Ni40NCAyNy4xOCIvPgogIDxwYXRoIGNsYXNzPSJzdDAiIGQ9Ik00Ni44NCwxMy44OHMuMy4zLjkuOXYzLjlzLS4zLjMtLjkuOWMwLDAtLjEtLjEtLjMtLjN2LTUuMmMtLjEuMSwwLDAsLjMtLjJaTTQ3LjQ0LDE5LjY4cy4xLjEuMy4zdjQuOXMtLjEuMS0uMy4zYzAsMC0uMy0uMy0uOS0uOXYtMy42Yy0uMS0uMS4zLS40LjktMVoiLz4KICA8cGF0aCBjbGFzcz0ic3QwIiBkPSJNNTEuMjQsMTkuNjhzLjMuMy45Ljl2My4ycy0uMy4zLS45LjljMCwwLS4xLS4xLS4zLS4zdi00LjVjMCwuMS4xLDAsLjMtLjJaTTUxLjg0LDEzLjg4aDQuNXMuMS4xLjMuM2MwLDAtLjMuMy0uOS45aC0zLjJzLS4zLS4zLS45LS45Yy0uMSwwLDAtLjEuMi0uM1pNNTIuMjQsMTguODhoMy43cy4yLjIuNi42aDBzLS4yLjItLjYuNmgtMy43cy0uMi0uMi0uNi0uNmgwYy0uMS4xLjEtLjIuNi0uNlpNNTIuNTQsMjMuODhoMy4ycy4zLjMuOS45YzAsMC0uMS4xLS4zLjNoLTQuNXMtLjEtLjEtLjMtLjNjMCwwLC4zLS4zLDEtLjlaTTU2Ljk0LDE0LjI4cy4xLjEuMy4zdjQuNXMtLjEuMS0uMy4zYzAsMC0uMy0uMy0uOS0uOXYtMy4yYzAtLjEuMy0uNC45LTFaIi8+CiAgPHBhdGggY2xhc3M9InN0MCIgZD0iTTYwLjY0LDEzLjg4aDQuNXMuMS4xLjMuM2MwLDAtLjMuMy0uOS45aC0zLjJzLS4zLS4zLS45LS45Yy0uMSwwLDAtLjEuMi0uM1pNNjEuMzQsMjMuODhoMy4ycy4zLjMuOS45YzAsMC0uMS4xLS4zLjNoLTQuNXMtLjEtLjEtLjMtLjNjMCwwLC4zLS4zLDEtLjlaTTYxLjA0LDE4Ljg4aDMuN3MuMi4yLjYuNmgwcy0uMi4yLS42LjZoLTMuN3MtLjItLjItLjYtLjZoMGwuNi0uNlpNNjUuNzQsMTQuMjhzLjEuMS4zLjN2NC41cy0uMS4xLS4zLjNjMCwwLS4zLS4zLS45LS45di0zLjJjMC0uMS4zLS40LjktMVpNNjUuNzQsMTkuNjhzLjEuMS4zLjN2NC41cy0uMS4xLS4zLjNjMCwwLS4zLS4zLS45LS45di0zLjJjMC0uMS4zLS40LjktMVoiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMTkuNTQsNDIuNThoLTcuNmMtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMwLC43LS41LDEuMi0xLjIsMS4yWiIvPgogIDxwYXRoIGNsYXNzPSJzdDIiIGQ9Ik0zMS45NCw0Mi41OGgtNy42Yy0uNywwLTEuMi0uNS0xLjItMS4ydi01LjFjMC0uNy41LTEuMiwxLjItMS4yaDcuNmMuNywwLDEuMi41LDEuMiwxLjJ2NS4xYzAsLjctLjUsMS4yLTEuMiwxLjJaIi8+CiAgPHBhdGggY2xhc3M9InN0MiIgZD0iTTQ0LjM0LDQyLjU4aC03LjZjLS43LDAtMS4yLS41LTEuMi0xLjJ2LTUuMWMwLS43LjUtMS4yLDEuMi0xLjJoNy42Yy43LDAsMS4yLjUsMS4yLDEuMnY1LjFjMCwuNy0uNSwxLjItMS4yLDEuMloiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNNTYuNzQsNDIuNThoLTcuNmMtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMuMS43LS41LDEuMi0xLjIsMS4yWiIvPgogIDxwYXRoIGNsYXNzPSJzdDYiIGQ9Ik02OS4yNCw0Mi41OGgtNy43Yy0uNywwLTEuMi0uNS0xLjItMS4ydi01LjFjMC0uNy41LTEuMiwxLjItMS4yaDcuNmMuNywwLDEuMi41LDEuMiwxLjJ2NS4xYy4xLjctLjUsMS4yLTEuMSwxLjJaIi8+CiAgPHBhdGggY2xhc3M9InN0MiIgZD0iTTE5LjU0LDUzLjg4aC03LjZjLS43LDAtMS4yLS41LTEuMi0xLjJ2LTUuMWMwLS43LjUtMS4yLDEuMi0xLjJoNy42Yy43LDAsMS4yLjUsMS4yLDEuMnY1LjFjMCwuNi0uNSwxLjItMS4yLDEuMloiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMzEuOTQsNTMuODhoLTcuNmMtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMwLC42LS41LDEuMi0xLjIsMS4yWiIvPgogIDxwYXRoIGNsYXNzPSJzdDIiIGQ9Ik00NC4zNCw1My44OGgtNy42Yy0uNywwLTEuMi0uNS0xLjItMS4ydi01LjFjMC0uNy41LTEuMiwxLjItMS4yaDcuNmMuNywwLDEuMi41LDEuMiwxLjJ2NS4xYzAsLjYtLjUsMS4yLTEuMiwxLjJaIi8+CiAgPHBhdGggY2xhc3M9InN0MiIgZD0iTTU2Ljc0LDUzLjg4aC03LjZjLS43LDAtMS4yLS41LTEuMi0xLjJ2LTUuMWMwLS43LjUtMS4yLDEuMi0xLjJoNy42Yy43LDAsMS4yLjUsMS4yLDEuMnY1LjFjLjEuNi0uNSwxLjItMS4yLDEuMloiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNNjkuMjQsNTMuODhoLTcuN2MtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMuMS42LS41LDEuMi0xLjEsMS4yWiIvPgogIDxwYXRoIGNsYXNzPSJzdDIiIGQ9Ik0xOS41NCw2NS4wOGgtNy42Yy0uNywwLTEuMi0uNS0xLjItMS4ydi01LjFjMC0uNy41LTEuMiwxLjItMS4yaDcuNmMuNywwLDEuMi41LDEuMiwxLjJ2NS4xYzAsLjctLjUsMS4yLTEuMiwxLjJaIi8+CiAgPHBhdGggY2xhc3M9InN0MiIgZD0iTTMxLjk0LDY1LjA4aC03LjZjLS43LDAtMS4yLS41LTEuMi0xLjJ2LTUuMWMwLS43LjUtMS4yLDEuMi0xLjJoNy42Yy43LDAsMS4yLjUsMS4yLDEuMnY1LjFjMCwuNy0uNSwxLjItMS4yLDEuMloiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNNDQuMzQsNjUuMDhoLTcuNmMtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMwLC43LS41LDEuMi0xLjIsMS4yWiIvPgogIDxwYXRoIGNsYXNzPSJzdDIiIGQ9Ik01Ni43NCw2NS4wOGgtNy42Yy0uNywwLTEuMi0uNS0xLjItMS4ydi01LjFjMC0uNy41LTEuMiwxLjItMS4yaDcuNmMuNywwLDEuMi41LDEuMiwxLjJ2NS4xYy4xLjctLjUsMS4yLTEuMiwxLjJaIi8+CiAgPHBhdGggY2xhc3M9InN0MiIgZD0iTTY5LjI0LDY1LjA4aC03LjdjLS43LDAtMS4yLS41LTEuMi0xLjJ2LTUuMWMwLS43LjUtMS4yLDEuMi0xLjJoNy42Yy43LDAsMS4yLjUsMS4yLDEuMnY1LjFjLjEuNy0uNSwxLjItMS4xLDEuMloiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMTkuNTQsNzYuMzhoLTcuNmMtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMwLC42LS41LDEuMi0xLjIsMS4yWiIvPgogIDxwYXRoIGNsYXNzPSJzdDIiIGQ9Ik0zMS45NCw3Ni4zOGgtNy42Yy0uNywwLTEuMi0uNS0xLjItMS4ydi01LjFjMC0uNy41LTEuMiwxLjItMS4yaDcuNmMuNywwLDEuMi41LDEuMiwxLjJ2NS4xYzAsLjYtLjUsMS4yLTEuMiwxLjJaIi8+CiAgPHBhdGggY2xhc3M9InN0MiIgZD0iTTQ0LjM0LDc2LjM4aC03LjZjLS43LDAtMS4yLS41LTEuMi0xLjJ2LTUuMWMwLS43LjUtMS4yLDEuMi0xLjJoNy42Yy43LDAsMS4yLjUsMS4yLDEuMnY1LjFjMCwuNi0uNSwxLjItMS4yLDEuMloiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNNTYuNzQsNzYuMzhoLTcuNmMtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMuMS42LS41LDEuMi0xLjIsMS4yWiIvPgogIDxwYXRoIGNsYXNzPSJzdDQiIGQ9Ik02OS4yNCw4Ny42OGgtNy43Yy0uNywwLTEuMi0uNS0xLjItMS4ydi0xNi40YzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjE2LjNjLjEuNy0uNSwxLjMtMS4xLDEuM1oiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMTkuNTQsODcuNjhoLTcuNmMtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMwLC42LS41LDEuMi0xLjIsMS4yWiIvPgogIDxwYXRoIGNsYXNzPSJzdDIiIGQ9Ik0zMS45NCw4Ny42OGgtNy42Yy0uNywwLTEuMi0uNS0xLjItMS4ydi01LjFjMC0uNy41LTEuMiwxLjItMS4yaDcuNmMuNywwLDEuMi41LDEuMiwxLjJ2NS4xYzAsLjYtLjUsMS4yLTEuMiwxLjJaIi8+CiAgPHBhdGggY2xhc3M9InN0MiIgZD0iTTQ0LjM0LDg3LjY4aC03LjZjLS43LDAtMS4yLS41LTEuMi0xLjJ2LTUuMWMwLS43LjUtMS4yLDEuMi0xLjJoNy42Yy43LDAsMS4yLjUsMS4yLDEuMnY1LjFjMCwuNi0uNSwxLjItMS4yLDEuMloiLz4KICA8cGF0aCBjbGFzcz0ic3QyIiBkPSJNNTYuNzQsODcuNjhoLTcuNmMtLjcsMC0xLjItLjUtMS4yLTEuMnYtNS4xYzAtLjcuNS0xLjIsMS4yLTEuMmg3LjZjLjcsMCwxLjIuNSwxLjIsMS4ydjUuMWMuMS42LS41LDEuMi0xLjIsMS4yWiIvPgo8L3N2Zz4=","width":26}
%---
%[control:editfield:42fa]
%   data: {"defaultValue":60,"label":"Time in seconds: ","run":"Nothing","valueType":"Double"}
%---
%[control:button:12c9]
%   data: {"label":"Convert to minutes","run":"Section"}
%---
%[control:dropdown:62b7]
%   data: {"defaultValue":"\"Total time\"","itemLabels":["Total time","Continuous time"],"items":["\"Total time\"","\"Continuous time\""],"label":"Approach for tracking time:","run":"Nothing"}
%---
%[control:editfield:630c]
%   data: {"defaultValue":1,"label":"# of Minutes:","run":"Nothing","valueType":"Double"}
%---
%[control:editfield:9037]
%   data: {"defaultValue":1.05,"label":"# of Minutes:","run":"Nothing","valueType":"Double"}
%---
%[control:editfield:1a62]
%   data: {"defaultValue":3,"label":"# of Minutes:","run":"Nothing","valueType":"Double"}
%---
%[control:editfield:42de]
%   data: {"defaultValue":10,"label":"# of Minutes:","run":"Nothing","valueType":"Double"}
%---
%[control:slider:235a]
%   data: {"defaultValue":25,"label":"First alert (%)","max":100,"min":0,"run":"Nothing","runOn":"ValueChanged","step":1}
%---
%[control:slider:7484]
%   data: {"defaultValue":50,"label":"Second alert (%)","max":100,"min":0,"run":"Nothing","runOn":"ValueChanged","step":1}
%---
%[control:slider:4a68]
%   data: {"defaultValue":75,"label":"Third alert (%)","max":100,"min":0,"run":"Nothing","runOn":"ValueChanged","step":1}
%---
%[control:slider:6949]
%   data: {"defaultValue":100,"label":"Fourth alert (%)","max":100,"min":0,"run":"Nothing","runOn":"ValueChanged","step":1}
%---
%[control:button:3a83]
%   data: {"label":"Submit","run":"Section"}
%---
%[control:editfield:0d78]
%   data: {"defaultValue":0,"label":"Enter a random number: ","run":"Nothing","valueType":"Double"}
%---
%[control:button:9ec4]
%   data: {"label":"Run simulation with no alerts","run":"Section"}
%---
%[control:button:6979]
%   data: {"label":"Run a simulation with alerts","run":"SectionToEnd"}
%---
