%[text] # An Algorithm for Continuous Text Neck Time
%%
%[text] ## Set Up Your Timer Algorithm
%[text] ![](text:image:180e) Use the calculator box below to convert from seconds to minutes (to the nearest hundredth). 
NumberOfSeconds = 0;     %[control:editfield:5fa3]{"position":[19,20]} %[control:button:559e]{"position":[24,25]}
TimeInMinutes = SecondsToMinutes(NumberOfSeconds);
function minutes = SecondsToMinutes(seconds)
minutes = seconds/60; % Convert seconds to minutes
end
if TimeInMinutes == round(TimeInMinutes,2)
    fprintf("%d seconds is %.2f minutes.",NumberOfSeconds,TimeInMinutes)
else
    fprintf("%d seconds is approximately %.2f minutes.",NumberOfSeconds,TimeInMinutes)
end
%%
clear Alert
%[text] **1. When the user enters text neck, wait** 
Alert.Time = 0.01*100; %[control:slider:1a2c]{"position":[14,18]}
%[text] **before alerting.**
%[text] 
%[text] **2. Alert the user for**
Alert.OffTime = 0.01*100; %[control:slider:6074]{"position":[17,21]}
%[text] 
%[text] **3. Before sending another alert, wait**
Alert.RestartTime = 0.01*100; %[control:slider:1773]{"position":[21,25]}

% Additional Alert settings
MaxTime = 1000;
Alert.History = false([1 MaxTime]);
Alert.TimeInTNP = false([1 MaxTime]);
Alert.Algorithm = "Continuous time";
%[text] 
%[text] Use the button below to submit your settings:
  %[control:button:71b4]{"position":[1,2]}
InputChecks = [CheckInputs(Alert.Time,MaxTime); CheckInputs(Alert.OffTime,MaxTime);CheckInputs(Alert.RestartTime,MaxTime)];
function Results = CheckInputs(Val,MaxVal)
if Val < 0
    NegVal = true;
    OverMaxVal = false;
elseif Val > MaxVal
    NegVal = false;
    OverMaxVal = true;
else
    NegVal = false;
    OverMaxVal = false;
end
Results = [NegVal OverMaxVal];
end
if any(InputChecks(:, 1))
    disp("Negative values do not make sense. Please use non-negative values.")
elseif any(InputChecks(:,2))
    disp("Please limit your alarm values to " + MaxTime/100 + " minutes.")
else
    disp("You set up the following alert algorithm:")
    fprintf("  Method: %s \n" + ...
        "  In text neck: %.2f minutes\n" + ...
        "  Alert stays on: %.2f minutes\n" + ...
        "  Between alerts: %.2f minutes\n",Alert.Algorithm,Alert.Time/100, ...
        Alert.OffTime/100,Alert.RestartTime/100)
end
%%
%[text] ## Test Your Algorithm
%%
%[text] #### User Scenario 2
  %[control:button:5db5]{"position":[1,2]}

% at a constant rate of 0.3 degrees per step
% From 0.4 to 1.4 minutes, move the head down, then stay there

if exist("Alert","var") && all(~InputChecks(:))

    [Alert,TextNeckState,TNPlot] = CreateScenario(Alert,40,140,"Down",0.3);

else
    warning("You must submit an algorithm before running a simulation.")
end
%%
%[text] #### User Scenario 4
  %[control:button:9eff]{"position":[1,2]}

if exist("Alert","var") && all(~InputChecks(:))

    [Alert,TextNeckState,TNPlot] = CreateScenario(Alert,...
        0,100,"Down",0.35,...
        100,199,"Up",0.3,...
        200,299,"Down",0.3,...
        300,399,"Up",0.3,...
        400,499,"Down",0.3,...
        500,599,"Up",0.3,...
        600,699,"Down",0.3,...
        700,799,"Up",0.3,...
        800,899,"Down",0.3,...
        900,999,"Up",0.3,...
        1000,1000,"Down",0.3);
else
    warning("You must submit an algorithm before running a simulation.")
end
%%
function TextNeckPlots = SetUpPlots(TotTimeSteps)
% SETUPPLOTS Function to set up 3D plots for torso and head, and a 2D plot for text neck over time
%
% Output Arguments:
%     TextNeckPlots - structure containing handles for the plots
%         .CurrPlot - handle for the current plot of text neck over time
%         .SensorPlot - handle for the sensor plot
%         .TextNeckAxes - axes for the text neck plot
%         .head - handle for the head plot
%         .torso - handle for the torso plot

clf % Clear current figure
f = figure('Position', [0 0 1100 400]); % Create a new figure with specified position
t = tiledlayout(1,2); % Create a tiled layout for multiple plots
nexttile; % Move to the next tile in the layout
% Plot torso as a 3D line
torso = plot3([4 4], [4 4], [4 8], 'SeriesIndex', 1, 'LineWidth', 3);
hold on % Hold the current plot to overlay additional plots
% Plot arms as a 3D line
arms = plot3([3 2.5 3 4 5 5.5 5], zeros(1, 7) + 4, [5 6 7 7 7 6 5], 'SeriesIndex', 1, 'LineWidth', 3);
% Plot legs as a 3D line
legs = plot3([3 3 3 4 5 5 5], [3.5, zeros(1, 5) + 4, 3.5], [1 1 3 4 3 1 1], 'SeriesIndex', 1, 'LineWidth', 3);
theta = linspace(0, 2*pi, 1000); % Create an array of angles for the head
x = 0.5 * cos(theta) + 4; % X coordinates for the head
y = zeros(1, length(theta)) + 4; % Y coordinates for the head
z = sin(theta) + 9; % Z coordinates for the head
% Plot head as a 3D line
head = plot3(x, y, z, 'SeriesIndex', 1, 'LineWidth', 3);
MyColors = colororder; % Get the current color order
% Fill areas with transparency to represent different sections
fill3([0 0 10 10], [0 10 10 0], [1 1 1 1], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
fill3([10 10 10 10], [0 10 10 0], [1 1 10 10], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
fill3([0 0 10 10], [10 10 10 10], [1 10 10 1], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
xlim([0 10]) % Set limits for the x-axis
ylim([0 10]) % Set limits for the y-axis
zlim([0 11]) % Set limits for the z-axis
hold off
ax = gca; % Get current axes
ax.Visible = 'off'; % Hide the axes
SensorPlot = PlotSensor(torso); % Call function to plot sensor data

% Set up the axes for the text neck plot
TextNeckAxes = axes(t);
TextNeckAxes.Layout.Tile = 2;   % Position the axes in the second tile
CurrPlot = plot(0, false, 'b'); % Initialize the current plot
xlim([0 TotTimeSteps]);         % Set limits of the text neck plot in 1/100 minute increments
ylim([-0.25 1.25]);             % Set limits of the text neck plot
xlabel('Minutes');              % Label for the x-axis
ylabel('Text Neck?');           % Label for the y-axis
title('Text Neck Over Time');   % Title for the plot
yticks([0 1]);                  % Set y-ticks
xticks(0:100:TotTimeSteps);     % Set x-ticks
xticklabels(0:1:10);            % Set x-tick labels
xtickangle(0);                  % Set angle for x-tick labels
yticklabels({'No', 'Yes'});     % Set y-tick labels

% Create output structure
TextNeckPlots.CurrPlot = CurrPlot;
TextNeckPlots.SensorPlot = SensorPlot;
TextNeckPlots.TextNeckAxes = TextNeckAxes;
TextNeckPlots.head = head;
TextNeckPlots.torso = torso;
end

function SensorPlot = PlotSensor(torso)

hold on

SensorPlot = scatter3(torso.XData(2), torso.YData(2), torso.ZData(2), 'SizeData', 30, 'SeriesIndex', 'none');

hold off

ax = gca;
ax.Visible = 'off';

end

function [TNPlot,TNData,Alert] = CheckPosition(i, TNPlot, TNData, Alert)
% CHECKPOSITION Function to check the position of the neck and update alerts
%
% Input Arguments:
%     i      - Current timestep index (0.01 minutes each)
%     TNPlot - Structure containing
%        CurrPlot - a line plot of the textneck status out/0 vs in/1
%        SensorPlot - a scatter showing where the sensor is located (at the
%              head/neck point on the figure)
%        TextNeckAxes - a handle to the axes containing CurrPlot
%        head - a handle to the line plot of the figure's head
%        torso - a handle to the line plot of the figure's torso
%     TNData - Structure containing data related to neck position
%        ContiguousTimeInTNP - the time in TextNeck without leaving
%        TextNeck - the current state, true is in TextNeck position
%        History - an in/out record for TextNeck position
%     Alert   - Structure containing alert timing parameters
%        Time - The amount of time in TextNeck that will set off an alarm
%        OffTime - The length of time an alarm will stay on while a user
%             persists in TextNeck position
%        RestartTime - The length of time after the alarm turns off before
%             the alarm will go off again while a user persists in TextNeck
%        History - an on/off record for the alerts
%
% Output Arguments:
%     TNPlot - Updated plot data structure
%     TNData - Updated data structure with TextNeck History
%     Alert  - Updated alert data structure with Alert History

% Update TNData value at time i

% First, calculate whether we are in TextNeck at all
TNData.TextNeck = DetermineNeckPosition(TNPlot);

TNData.History(i) = double(TNData.TextNeck);

% Run algorithm to determine whether an alarm should be going off
[Alert,TNData] = ShouldAlarm(i,TNData,Alert);

% Update current plot data with new index and TextNeck status
TNPlot = UpdatePlotsWithAlarms(i,TNPlot,TNData,Alert);
end

function TNData = InitializeTextNeckData(TotTimeSteps)
TNData.ContiguousTimeInTNP = 0;
TNData.TextNeck = false;
TNData.History = zeros([1 TotTimeSteps]);
end

function [Alert,TNState,TNPlot] = CreateScenario(Alert,Start,Stop,Mode,Speed)
arguments
    Alert
end
arguments (Repeating)
    Start (1,1) double {mustBeInteger}
    Stop (1,1) double {mustBeInteger}
    Mode (1,1) string {mustBeMember(Mode,["Up","Down"])}
    Speed (1,1) double
end
% Note that Start and Stop are timestep indices, not times
TotTimeSteps = 1000; % With step size 0.01 min
TNPlot = SetUpPlots(TotTimeSteps);
TNData = InitializeTextNeckData(TotTimeSteps);

TopOfSpine = [TNPlot.torso.XData(2), TNPlot.torso.YData(2), TNPlot.torso.ZData(2)];
MoveIdx = 1;

if any([Stop{:}]-[Start{:}]<0)
    error("Wrong direction!")
    return
end

for i = 1:TotTimeSteps % in 0.01 minute increments
    if MoveIdx <= length(Start)
        if i == Stop{MoveIdx}
            MoveIdx = MoveIdx + 1; % Move to the next scenario
        elseif (i >= Start{MoveIdx}) && (i < Stop{MoveIdx})
            % Move the head down or up based on the mode
            if Mode{MoveIdx} == "Down"
                rotate(TNPlot.head, [1 0 0], Speed{MoveIdx}, TopOfSpine);
            else % Move up
                rotate(TNPlot.head, [1 0 0], -Speed{MoveIdx}, TopOfSpine);
            end
        end
    end
    [TNPlot,TNData,Alert] = CheckPosition(i, TNPlot, TNData, Alert);
end
TNState = TNData.History;
if all(~Alert.History)
    disp("The wearable alert system did not alarm in this scenario.");
end
end

function TextNeck = DetermineNeckPosition(TNPlot,TNAngle)
arguments
    TNPlot
    TNAngle (1,1) double {mustBeInteger,mustBePositive} = 20
end
% Calculate vectors from torso to the top of the head
% 250/1000 is one quarter of the way around
% and between torso points to compute the neck angle
Vec1 = [TNPlot.head.XData(250) TNPlot.head.YData(250) TNPlot.head.ZData(250)] - [TNPlot.torso.XData(2) TNPlot.torso.YData(2) TNPlot.torso.ZData(2)];
Vec2 = [TNPlot.torso.XData(1) TNPlot.torso.YData(1) TNPlot.torso.ZData(1)] - [TNPlot.torso.XData(2) TNPlot.torso.YData(2) TNPlot.torso.ZData(2)];

% Compute the angle of the neck in degrees
% Note that standing straight is a neck angle of 180
angle = atan2d(norm(cross(Vec1,Vec2)), dot(Vec1,Vec2));

% Compute TextNeck values for the current position
if (angle > (180+TNAngle)) || (angle < (180-TNAngle))
    % Set TextNeck=true if the neck is bent more than 20 degrees
    TextNeck = true;
    % If the neck angle moves within 20 degrees of 180
elseif (angle <= (180+TNAngle)) && (angle >= (180-TNAngle))
    TextNeck = false;
end
end

function [Alert,TNData] = ShouldAlarm(timeStep,TNData,Alert)
if Alert.Algorithm == "Total time"
    % The Total Time Algorithm says that we should alert if
    %    In TextNeck position AND (sum(TNData.History) > Alert.Time)
    %       AND (mod(sum(TNData.History)-Alert.Time,RestartTime+OffTime) < OffTime) 

    % Check whether an alert is triggered
    if TNData.TextNeck && sum(TNData.History) >= Alert.Time && ...
            (mod(sum(TNData.History)-Alert.Time,Alert.OffTime+Alert.RestartTime) < Alert.OffTime)
        Alert.History(timeStep) = true;
    else
        Alert.History(timeStep) = false;
    end
elseif Alert.Algorithm == "Continuous time"
    % Increment contiguous time in TextNeck if currently in that state or reset
    % it to 0 if we are not currently in TextNeck
    if TNData.TextNeck == true
        TNData.ContiguousTimeInTNP = TNData.ContiguousTimeInTNP + 1;
    else
        TNData.ContiguousTimeInTNP = 0;
    end
    
    % The Continuous Time Algorithm says that we should alert if
    %    In TextNeck position AND (ContiguousTimeInTNP > Alert.Time)
    %       AND (mod(ContiguousTimeInTNP-Alert.Time,RestartTime+OffTime) < OffTime)       [TimeAlertOn was TimeSinceAlert]
    %         OR (TimeAlertOff > RestartTime)) [TimeAlertOff was Resting]

    % Check whether an alert is triggered
    if TNData.TextNeck && TNData.ContiguousTimeInTNP >= Alert.Time && ...
            (mod(TNData.ContiguousTimeInTNP-Alert.Time,Alert.OffTime+Alert.RestartTime) < Alert.OffTime)
        Alert.History(timeStep) = true;
    else
        Alert.History(timeStep) = false;
    end
else
    error("Unexpected algorithm selection: " + ...
        Alert.Algorithm)
end
end

function TNPlot = UpdatePlotsWithAlarms(timeStep,TNPlot,TNData,Alert)
% Update current plot data with new index and TextNeck status
if Alert.History(timeStep)
    % Turn the sensor icon on
    TNPlot.SensorPlot.MarkerFaceColor = 'r';
    % Extend red shading over this timestep
    hold on
    TNPlot.CurrPlot = plot(timeStep-1, TNPlot.CurrPlot.YData(end), 'r', 'Parent', TNPlot.TextNeckAxes);
    patch([timeStep-1 timeStep-1 timeStep timeStep], [-0.25 1.25 1.25 -0.25], 'red', 'EdgeColor', 'none', 'FaceAlpha', 0.2, 'Parent', TNPlot.TextNeckAxes);
    hold off
else
    % Turn the sensor icon off
    TNPlot.SensorPlot.MarkerFaceColor = 'none';
    % Switch line back to blue
    hold on
    TNPlot.CurrPlot = plot(timeStep-1, TNPlot.CurrPlot.YData(end), 'b', 'Parent', TNPlot.TextNeckAxes);
    hold off
end
TNPlot.CurrPlot.XData = [TNPlot.CurrPlot.XData timeStep];
TNPlot.CurrPlot.YData = [TNPlot.CurrPlot.YData TNData.History(timeStep)];

% Ensure the plot is updated with the latest data
if mod(timeStep,10) == 0
    pause(0.05);
end
end

%[appendix]{"version":"1.0"}
%---
%[metadata:view]
%   data: {"layout":"hidecode","rightPanelPercent":40}
%---
%[text:image:180e]
%   data: {"align":"baseline","height":31,"src":"data:image\/svg+xml;base64,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","width":26}
%---
%[control:editfield:5fa3]
%   data: {"defaultValue":0,"label":"Time in seconds: ","run":"Nothing","valueType":"Double"}
%---
%[control:button:559e]
%   data: {"label":"Convert to minutes","run":"Section"}
%---
%[control:slider:1a2c]
%   data: {"defaultValue":0.01,"label":"# of Minutes:","max":10,"min":0,"run":"Nothing","runOn":"ValueChanging","step":0.01}
%---
%[control:slider:6074]
%   data: {"defaultValue":0.01,"label":"# of Minutes:","max":10,"min":0,"run":"Nothing","runOn":"ValueChanging","step":0.01}
%---
%[control:slider:1773]
%   data: {"defaultValue":0.01,"label":"# of Minutes:","max":10,"min":0,"run":"Nothing","runOn":"ValueChanging","step":0.01}
%---
%[control:button:71b4]
%   data: {"label":"Submit","run":"Section"}
%---
%[control:button:5db5]
%   data: {"label":"Run scenario 2","run":"Section"}
%---
%[control:button:9eff]
%   data: {"label":"Run scenario 4","run":"Section"}
%---
